<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    protected $signatureService;

    public function __construct(\App\Services\SignatureService $signatureService)
    {
        $this->signatureService = $signatureService;
    }

    public function create(Request $request)
    {
        $validated = $request->validate([
            'reference_id' => 'required|string',
            'amount' => 'required|numeric|min:1',
            'currency' => 'nullable|string|size:3',
            'customer_name' => 'nullable|string',
            'customer_email' => 'nullable|email',
            'customer_phone' => 'nullable|string',
            'success_url' => 'required|url',
            'fail_url' => 'required|url',
        ]);

        $site = $request->attributes->get('site'); // Set by middleware

        // Initialize Razorpay API
        // For now using placeholders, will need env config
        $api = new \Razorpay\Api\Api(config('services.razorpay.key'), config('services.razorpay.secret'));

        $orderData = [
            'receipt' => $validated['reference_id'],
            'amount' => $validated['amount'] * 100, // INR in paisa
            'currency' => $validated['currency'] ?? 'INR',
            'payment_capture' => 1 // Auto capture
        ];

        try {
            $razorpayOrder = $api->order->create($orderData);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Razorpay Order Creation Failed: ' . $e->getMessage()], 500);
        }

        $payment = $site->payments()->create([
            'reference_id' => $validated['reference_id'],
            'razorpay_order_id' => $razorpayOrder['id'],
            'amount' => $validated['amount'],
            'currency' => $validated['currency'] ?? 'INR',
            'status' => 'pending',
            'customer_email' => $validated['customer_email'],
            'customer_phone' => $validated['customer_phone'],
            'success_url' => $validated['success_url'],
            'fail_url' => $validated['fail_url'],
        ]);

        return response()->json([
            'status' => 'success',
            'payment_url' => route('checkout.show', ['payment' => $payment->id]),
            'payment_id' => $payment->id,
        ]);
    }

    public function status(Request $request)
    {
        $request->validate(['reference_id' => 'required|string']);
        $site = $request->attributes->get('site'); // Set by middleware

        $payment = $site->payments()->where('reference_id', $request->reference_id)->first();

        if (!$payment) {
            return response()->json(['status' => 'error', 'message' => 'Payment not found'], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'reference_id' => $payment->reference_id,
                'status' => $payment->status,
                'amount' => $payment->amount,
                'currency' => $payment->currency,
                'razorpay_payment_id' => $payment->razorpay_payment_id,
            ]
        ]);
    }
}
